<?php
/**
 * Plugin Name: SMM API Integration
 * Description: دمج API خارجي للطلبات والاستعلام عن الحالة عبر شورت كود.
 * Version: 1.0.0
 * Author: You
 */

if (!defined('ABSPATH')) exit;

class SMM_API_Integration {
    const OPTION_KEY = 'smm_api_settings';

    public function __construct() {
        add_action('admin_menu', [$this, 'add_settings_page']);
        add_action('admin_init', [$this, 'register_settings']);

        add_shortcode('smm_order_form', [$this, 'shortcode_order_form']);
        add_shortcode('smm_order_status', [$this, 'shortcode_order_status']);

        add_action('wp_enqueue_scripts', [$this, 'enqueue_assets']);
        add_action('wp_ajax_smm_place_order', [$this, 'handle_place_order']);
        add_action('wp_ajax_nopriv_smm_place_order', [$this, 'handle_place_order']);
        add_action('wp_ajax_smm_check_status', [$this, 'handle_check_status']);
        add_action('wp_ajax_nopriv_smm_check_status', [$this, 'handle_check_status']);
    }

    /** ---------- Admin Settings ---------- */
    public function add_settings_page() {
        add_options_page('SMM API', 'SMM API', 'manage_options', 'smm-api', [$this, 'settings_page_html']);
    }

    public function register_settings() {
        register_setting(self::OPTION_KEY, self::OPTION_KEY, [
            'type' => 'array',
            'sanitize_callback' => function($input){
                return [
                    'api_url' => esc_url_raw($input['api_url'] ?? ''),
                    'api_key' => trim($input['api_key'] ?? ''),
                ];
            },
        ]);

        add_settings_section('smm_api_main', 'إعدادات API', '__return_false', self::OPTION_KEY);

        add_settings_field('api_url', 'API URL', function(){
            $opt = get_option(self::OPTION_KEY);
            printf('<input type="url" name="%s[api_url]" value="%s" class="regular-text" placeholder="https://example.com/api/v2" required>',
                esc_attr(self::OPTION_KEY),
                isset($opt['api_url']) ? esc_attr($opt['api_url']) : ''
            );
        }, self::OPTION_KEY, 'smm_api_main');

        add_settings_field('api_key', 'API Key', function(){
            $opt = get_option(self::OPTION_KEY);
            printf('<input type="text" name="%s[api_key]" value="%s" class="regular-text" placeholder="sk_live_xxx" required>',
                esc_attr(self::OPTION_KEY),
                isset($opt['api_key']) ? esc_attr($opt['api_key']) : ''
            );
        }, self::OPTION_KEY, 'smm_api_main');
    }

    public function settings_page_html() {
        if (!current_user_can('manage_options')) return;
        ?>
        <div class="wrap">
            <h1>إعدادات SMM API</h1>
            <form method="post" action="options.php">
                <?php
                settings_fields(self::OPTION_KEY);
                do_settings_sections(self::OPTION_KEY);
                submit_button();
                ?>
            </form>
        </div>
        <?php
    }

    /** ---------- Client: API Wrapper ---------- */
    private function api_request($body) {
        $opt = get_option(self::OPTION_KEY);
        $api_url = $opt['api_url'] ?? '';
        $api_key = $opt['api_key'] ?? '';

        if (empty($api_url) || empty($api_key)) {
            return new WP_Error('smm_missing_config', 'الرجاء ضبط API URL و API Key من الإعدادات.');
        }

        $body = array_merge(['key' => $api_key], $body);

        $resp = wp_remote_post($api_url, [
            'timeout' => 30,
            'headers' => [
                'Content-Type' => 'application/x-www-form-urlencoded; charset=utf-8',
                'User-Agent'   => 'WP-SMM-Client/1.0',
            ],
            'sslverify' => true, // مهم: لا توقف التحقق من SSL
            'body' => $body,
        ]);

        if (is_wp_error($resp)) return $resp;

        $code = wp_remote_retrieve_response_code($resp);
        $body = wp_remote_retrieve_body($resp);

        if ($code < 200 || $code >= 300) {
            return new WP_Error('smm_bad_status', 'فشل الطلب إلى API: HTTP ' . $code, ['body' => $body]);
        }

        $json = json_decode($body, true);
        if (json_last_error() !== JSON_ERROR_NONE) {
            return new WP_Error('smm_json_error', 'استجابة غير صالحة من API.');
        }
        return $json;
    }

    private function api_services() {
        return $this->api_request(['action' => 'services']);
    }

    private function api_order($data) {
        return $this->api_request(array_merge(['action' => 'add'], $data));
    }

    private function api_status($order_id) {
        return $this->api_request(['action' => 'status', 'order' => $order_id]);
    }

    /** ---------- Frontend: Assets ---------- */
    public function enqueue_assets() {
        wp_register_script('smm-api', plugin_dir_url(__FILE__) . 'smm-api.js', ['jquery'], '1.0', true);
        wp_localize_script('smm-api', 'SMM_API', [
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce'    => wp_create_nonce('smm_api_nonce')
        ]);
    }

    /** ---------- Shortcodes ---------- */
    public function shortcode_order_form($atts) {
        // بإمكانك تمرير service عبر الشورتكود: [smm_order_form service="1"]
        $atts = shortcode_atts(['service' => ''], $atts, 'smm_order_form');
        wp_enqueue_script('smm-api');

        ob_start(); ?>
        <form id="smm-order-form">
            <input type="hidden" name="action" value="smm_place_order">
            <input type="hidden" name="nonce" value="<?php echo esc_attr(wp_create_nonce('smm_api_nonce')); ?>">
            <div>
                <label>Service ID</label>
                <input type="number" name="service" required value="<?php echo esc_attr($atts['service']); ?>">
            </div>
            <div>
                <label>Link</label>
                <input type="url" name="link" required placeholder="http://example.com/test">
            </div>
            <div>
                <label>Quantity (اختياري)</label>
                <input type="number" name="quantity" min="1">
            </div>
            <button type="submit">إرسال الطلب</button>
            <div id="smm-order-result" style="margin-top:8px;"></div>
        </form>
        <?php
        return ob_get_clean();
    }

    public function shortcode_order_status() {
        wp_enqueue_script('smm-api');
        ob_start(); ?>
        <form id="smm-status-form">
            <input type="hidden" name="action" value="smm_check_status">
            <input type="hidden" name="nonce" value="<?php echo esc_attr(wp_create_nonce('smm_api_nonce')); ?>">
            <div>
                <label>Order ID</label>
                <input type="number" name="order_id" required>
            </div>
            <button type="submit">تحقق من الحالة</button>
            <div id="smm-status-result" style="margin-top:8px;"></div>
        </form>
        <?php
        return ob_get_clean();
    }

    /** ---------- AJAX Handlers ---------- */
    public function handle_place_order() {
        check_ajax_referer('smm_api_nonce', 'nonce');

        $service  = isset($_POST['service']) ? intval($_POST['service']) : 0;
        $link     = isset($_POST['link']) ? esc_url_raw($_POST['link']) : '';
        $quantity = isset($_POST['quantity']) ? intval($_POST['quantity']) : null;

        if (!$service || empty($link)) {
            wp_send_json_error(['message' => 'الرجاء تعبئة الحقول المطلوبة.']);
        }

        $payload = ['service' => $service, 'link' => $link];
        if (!empty($quantity)) $payload['quantity'] = $quantity;

        $res = $this->api_order($payload);
        if (is_wp_error($res)) {
            wp_send_json_error(['message' => $res->get_error_message()]);
        }
        wp_send_json_success($res);
    }

    public function handle_check_status() {
        check_ajax_referer('smm_api_nonce', 'nonce');

        $order_id = isset($_POST['order_id']) ? intval($_POST['order_id']) : 0;
        if (!$order_id) {
            wp_send_json_error(['message' => 'رقم الطلب غير صالح.']);
        }

        $res = $this->api_status($order_id);
        if (is_wp_error($res)) {
            wp_send_json_error(['message' => $res->get_error_message()]);
        }
        wp_send_json_success($res);
    }
}

new SMM_API_Integration();
