<?php
if ( ! defined( 'ABSPATH' ) ) exit;

/**
 * Class SMM_MP_Metabox
 * Adds provider & service fields to products, shows fields in frontend, saves item meta to order items,
 * and admin metabox for order status checks.
 */
class SMM_MP_Metabox {

    const META_PROVIDER = '_smm_provider_id';
    const META_SERVICE  = '_smm_service_id';
    const CART_KEY_LINK = 'smm_link';
    const CART_KEY_QTY  = 'smm_quantity';
    const ITEM_META_API_ORDER = 'smm_api_order_id';

    public function __construct() {
        // product metabox fields
        add_action( 'woocommerce_product_options_general_product_data', [ $this, 'product_fields' ] );
        add_action( 'woocommerce_process_product_meta', [ $this, 'save_product_fields' ] );

        // frontend fields (link + qty)
        add_action( 'woocommerce_before_add_to_cart_button', [ $this, 'render_frontend_fields' ] );
        add_filter( 'woocommerce_add_to_cart_validation', [ $this, 'validate_frontend_fields' ], 10, 3 );
        add_filter( 'woocommerce_add_cart_item_data', [ $this, 'save_cart_item_data' ], 10, 2 );
        add_filter( 'woocommerce_get_item_data', [ $this, 'display_cart_item_data' ], 10, 2 );

        // add order item meta during checkout
        add_action( 'woocommerce_checkout_create_order_line_item', [ $this, 'add_to_order_items' ], 10, 4 );

        // admin order metabox (status checking)
        add_action( 'add_meta_boxes', [ $this, 'add_order_metabox' ] );
    }

    public function product_fields() {
        $providers = SMM_MP_Provider::get_active_providers();
        echo '<div class="options_group">';
        woocommerce_wp_select( [
            'id' => self::META_PROVIDER,
            'label' => 'اختر المزود',
            'options' => $this->get_provider_options( $providers ),
            'desc_tip' => true,
            'description' => 'اختر مزوّد الخدمة لهذا المنتج (إذا لم تختَر فلن يُرسل إلى SMM).'
        ] );
        woocommerce_wp_text_input( [
            'id' => self::META_SERVICE,
            'label' => 'Service ID',
            'desc_tip' => true,
            'description' => 'ضع رقم الخدمة لدى المزود.',
            'type' => 'number'
        ] );
        echo '</div>';
    }

    private function get_provider_options( $providers ) {
        $out = [ '' => '— اختر مزوّد —' ];
        foreach ( $providers as $p ) {
            $out[ $p->ID ] = $p->post_title;
        }
        return $out;
    }

    public function save_product_fields( $post_id ) {
        if ( isset( $_POST[ self::META_PROVIDER ] ) ) {
            update_post_meta( $post_id, self::META_PROVIDER, absint( $_POST[ self::META_PROVIDER ] ) );
        }
        if ( isset( $_POST[ self::META_SERVICE ] ) ) {
            update_post_meta( $post_id, self::META_SERVICE, sanitize_text_field( $_POST[ self::META_SERVICE ] ) );
        }
    }

    /* ---------- Frontend fields ---------- */

    public function render_frontend_fields() {
        global $product;
        if ( ! $product ) return;
        $product_id = $product->get_id();
        $provider = get_post_meta( $product_id, self::META_PROVIDER, true );
        $service  = get_post_meta( $product_id, self::META_SERVICE, true );
        if ( ! $provider || ! $service ) return;

        ?>
        <div class="smm-fields" style="margin:12px 0;">
            <p>
                <label for="smm_link"><strong>الرابط المطلوب</strong></label><br>
                <input type="url" id="smm_link" name="smm_link" required placeholder="https://example.com/test" style="width:100%;">
            </p>
            <p>
                <label for="smm_quantity"><strong>الكمية (API)</strong></label><br>
                <input type="number" id="smm_quantity" name="smm_quantity" required min="1" step="1" style="width:150px;">
            </p>
        </div>
        <?php
    }

    public function validate_frontend_fields( $passed, $product_id, $quantity ) {
        $provider = get_post_meta( $product_id, self::META_PROVIDER, true );
        $service  = get_post_meta( $product_id, self::META_SERVICE, true );
        if ( ! $provider || ! $service ) return $passed;

        $link = isset( $_POST['smm_link'] ) ? esc_url_raw( $_POST['smm_link'] ) : '';
        $qty  = isset( $_POST['smm_quantity'] ) ? absint( $_POST['smm_quantity'] ) : 0;

        if ( empty( $link ) ) {
            wc_add_notice( 'الرجاء إدخال الرابط للمنتج.', 'error' );
            $passed = false;
        }
        if ( $qty < 1 ) {
            wc_add_notice( 'الرجاء إدخال كمية صحيحة للـAPI.', 'error' );
            $passed = false;
        }
        return $passed;
    }

    public function save_cart_item_data( $cart_item_data, $product_id ) {
        $provider = get_post_meta( $product_id, self::META_PROVIDER, true );
        $service  = get_post_meta( $product_id, self::META_SERVICE, true );
        if ( ! $provider || ! $service ) return $cart_item_data;

        if ( isset( $_POST['smm_link'] ) ) {
            $cart_item_data[ self::CART_KEY_LINK ] = esc_url_raw( $_POST['smm_link'] );
        }
        if ( isset( $_POST['smm_quantity'] ) ) {
            $cart_item_data[ self::CART_KEY_QTY ] = absint( $_POST['smm_quantity'] );
        }
        // store product provider/service for clarity
        $cart_item_data[ self::META_PROVIDER ] = absint( $provider );
        $cart_item_data[ self::META_SERVICE ]  = sanitize_text_field( $service );

        // unique key so items with diff fields stack separately
        $cart_item_data['unique_key'] = md5( maybe_serialize( [ $cart_item_data, time() ] ) );

        return $cart_item_data;
    }

    public function display_cart_item_data( $item_data, $cart_item ) {
        if ( isset( $cart_item[ self::CART_KEY_LINK ] ) ) {
            $item_data[] = [
                'name' => 'الرابط',
                'value' => esc_url( $cart_item[ self::CART_KEY_LINK ] ),
            ];
        }
        if ( isset( $cart_item[ self::CART_KEY_QTY ] ) ) {
            $item_data[] = [
                'name' => 'الكمية (API)',
                'value' => intval( $cart_item[ self::CART_KEY_QTY ] ),
            ];
        }
        return $item_data;
    }

    public function add_to_order_items( $item, $cart_item_key, $values, $order ) {
        if ( isset( $values[ self::CART_KEY_LINK ] ) ) {
            $item->add_meta_data( 'SMM Link', esc_url_raw( $values[ self::CART_KEY_LINK ] ), true );
        }
        if ( isset( $values[ self::CART_KEY_QTY ] ) ) {
            $item->add_meta_data( 'SMM Quantity', absint( $values[ self::CART_KEY_QTY ] ), true );
        }
        if ( isset( $values[ self::META_PROVIDER ] ) ) {
            $item->add_meta_data( 'SMM Provider ID', absint( $values[ self::META_PROVIDER ] ), true );
        }
        if ( isset( $values[ self::META_SERVICE ] ) ) {
            $item->add_meta_data( 'SMM Service ID', sanitize_text_field( $values[ self::META_SERVICE ] ), true );
        }
    }

    /* ---------- Admin Order Metabox (status buttons) ---------- */

    public function add_order_metabox() {
        add_meta_box( 'smm_order_status_box', 'SMM — حالة الطلبات الخارجية', [ $this, 'render_order_status_box' ], 'shop_order', 'side', 'default' );
    }

    public function render_order_status_box( $post ) {
        $order = wc_get_order( $post->ID );
        if ( ! $order ) {
            echo '<p>لا يوجد طلب.</p>';
            return;
        }

        $items = $order->get_items();
        $has = false;
        echo '<div id="smm-order-status-list">';
        foreach ( $items as $item_id => $item ) {
            $api_order = $item->get_meta( 'SMM API Order ID' );
            if ( $api_order ) {
                $has = true;
                echo '<div style="margin-bottom:10px;">';
                echo '<div style="font-weight:600;">' . esc_html( $item->get_name() ) . '</div>';
                echo '<div>API Order ID: <code>' . esc_html( $api_order ) . '</code></div>';
                echo '<button class="button smm-mp-check-one" data-item-id="'.esc_attr($item_id).'" data-order-id="'.esc_attr($post->ID).'">تحقق من الحالة</button>';
                echo '<div class="smm-mp-status-output" id="smm-mp-status-'.$item_id.'" style="margin-top:6px;font-size:12px;"></div>';
                echo '</div>';
            }
        }
        echo '</div>';
        if ( ! $has ) {
            echo '<p>لا توجد عناصر مرتبطة بطلب خارجي بعد.</p>';
        } else {
            echo '<hr />';
            echo '<button class="button button-primary smm-mp-check-all" data-order-id="'.esc_attr($post->ID).'">تحقق من حالة جميع العناصر</button>';
            echo '<div id="smm-mp-status-all" style="margin-top:8px;font-size:12px;"></div>';
        }
        // nonce for ajax
        wp_nonce_field( 'smm_mp_nonce', 'smm_mp_nonce_field' );
    }
}
