<?php

namespace Hostinger;

use Hostinger\Admin\PluginSettings;

defined( 'ABSPATH' ) || exit;

class LlmsTxtGenerator {

	public const HOSTINGER_LLMSTXT_FILENAME  = 'llms.txt';
	public const HOSTINGER_LLMSTXT_SIGNATURE = '[comment]: # (Generated by Hostinger Tools Plugin)';

	protected const HOSTINGER_LLMSTXT_SUPPORTED_POST_TYPES = array(
		'post',
		'page',
	);
	protected const UTF8_BOM                               = "\xEF\xBB\xBF";

	protected PluginSettings $plugin_settings;


	public function __construct( PluginSettings $plugin_settings ) {
		$this->plugin_settings = $plugin_settings;
		add_action( 'init', array( $this, 'init' ) );
	}

	public function init(): void {
		if ( wp_doing_ajax() || wp_doing_cron() || ! current_user_can( 'manage_options' ) ) {
			return;
		}

		$settings = $this->plugin_settings->get_plugin_settings();

		if ( $settings->get_enable_llms_txt() && ! $this->llmstxt_file_exists() ) {
			$this->generate();
		}

		$this->init_hooks();
	}

	public function on_settings_update( bool $is_enabled ): void {
		if ( $is_enabled ) {
			$this->generate();
		} else {
			$this->delete();
		}
	}

	public function is_user_generated_file(): bool {
		if ( ! $this->llmstxt_file_exists() ) {
			return false;
		}

		global $wp_filesystem;
		$this->init_wp_filesystem();
		$content = $wp_filesystem->get_contents( $this->get_llmstxt_file_path() );

		if ( $content === false ) {
			return false;
		}

		return ! str_contains( $content, self::HOSTINGER_LLMSTXT_SIGNATURE );
	}

	public function on_post_status_change( string $new_status, string $old_status, \WP_Post $post ): void {
		if ( ! $this->is_post_type_supported( $post->post_type ) ) {
			return;
		}

		if ( $new_status === 'publish' || $old_status === 'publish' ) {
			$this->generate();
		}
	}

	public function on_blog_change( mixed $old_value, mixed $new_value ): void {
		if ( $old_value !== $new_value ) {
			$this->generate();
		}
	}

	public function get_content(): string {
		$content  = self::UTF8_BOM;
		$content .= $this->inject_title();
		$content .= $this->inject_site_description();
		$content .= $this->inject_items( $this->get_by_post_type( 'post' ), 'Posts' );
		$content .= $this->inject_items( $this->get_by_post_type( 'page' ), 'Pages' );
		$content .= $this->inject_signature();

		return $content;
	}

	public function init_hooks(): void {
		add_action( 'transition_post_status', array( $this, 'on_post_status_change' ), 10, 3 );
		add_action( 'hostinger_tools_setting_enable_llms_txt_update', array( $this, 'on_settings_update' ) );
		add_action( 'update_option_blogname', array( $this, 'on_blog_change' ), 10, 2 );
		add_action( 'update_option_blogdescription', array( $this, 'on_blog_change' ), 10, 2 );
	}

	protected function generate(): void {
		if ( $this->is_user_generated_file() ) {
			return;
		}

		global $wp_filesystem;
		$this->init_wp_filesystem();
		$wp_filesystem->put_contents( $this->get_llmstxt_file_path(), $this->get_content() );
	}

	protected function delete(): void {
		if ( $this->llmstxt_file_exists() && ! $this->is_user_generated_file() ) {
			global $wp_filesystem;
			$this->init_wp_filesystem();
			$wp_filesystem->delete( $this->get_llmstxt_file_path() );
		}
	}

	protected function get_llmstxt_file_path(): string {
		return ABSPATH . self::HOSTINGER_LLMSTXT_FILENAME;
	}

	protected function llmstxt_file_exists(): bool {
		return file_exists( $this->get_llmstxt_file_path() );
	}

	protected function is_post_type_supported( string $post_type ): bool {
		return in_array( $post_type, self::HOSTINGER_LLMSTXT_SUPPORTED_POST_TYPES, true );
	}

	protected function get_by_post_type( string $post_type, int $limit = 100 ): array {
		$args = array(
			'post_type'      => $post_type,
			'post_status'    => 'publish',
			'fields'         => 'ids',
			'posts_per_page' => apply_filters( 'hostinger_llmstext_item_limit', $limit, $post_type ),
		);

		return get_posts( $args );
	}

	protected function inject_site_description(): string {
		$description = get_bloginfo( 'description' );
		return $description ? "> $description\n\n" : '';
	}

	protected function inject_title(): string {
		$title = get_bloginfo( 'name' ) ?  get_bloginfo( 'name' ) : site_url();
		return "# $title\n\n";
	}

	protected function inject_signature(): string {
		return "\n\n" . self::HOSTINGER_LLMSTXT_SIGNATURE;
	}

	protected function inject_items( array $items, string $title ): string {
		if ( empty( $items ) ) {
			return '';
		}

		$content = "\n## $title\n\n";

		foreach ( $items as $item ) {
			$post      = get_post( $item );
			$title     = $post->post_title;
			$permalink = get_permalink( $post );
			$excerpt   = $this->prepare_excerpt( $post );

			$content .= "- [$title]($permalink)";
			if ( $excerpt ) {
				$content .= ": $excerpt";
			}

			$content .= "\n";
		}

		return $content;
	}

	protected function prepare_excerpt( \WP_Post $item ): string {
		return html_entity_decode( wp_trim_excerpt( $item->post_excerpt, $item ) );
	}

	protected function init_wp_filesystem() {
		global $wp_filesystem;
		require_once ABSPATH . '/wp-admin/includes/file.php';
		WP_Filesystem();
	}
}
