<?php
/**
 * Plugin Name: SMM Multi Provider + WooCommerce
 * Description: دعم عدة مزوّدين SMM، ربط منتجات WooCommerce بالمزود/Service ID، تزامن أسعار وخدمة تحديث حالات الطلبات تلقائياً.
 * Version: 1.0.0
 * Author: You
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

define( 'SMM_MP_PLUGIN_DIR', plugin_dir_path( __FILE__ ) );
define( 'SMM_MP_PLUGIN_URL', plugin_dir_url( __FILE__ ) );

/**
 * Autoload includes (simple)
 */
require_once SMM_MP_PLUGIN_DIR . 'includes/class-smm-provider.php';
require_once SMM_MP_PLUGIN_DIR . 'includes/class-smm-metabox.php';
require_once SMM_MP_PLUGIN_DIR . 'includes/class-smm-sync.php';

/**
 * Init classes
 */
add_action( 'init', function() {
    SMM_MP_Provider::register_post_type();
} );

add_action( 'plugins_loaded', function() {
    // instantiate admin/product metaboxes and sync handlers
    new SMM_MP_Metabox();
    new SMM_MP_Sync();
} );

/**
 * Activation / Deactivation: schedule cron events
 */
register_activation_hook( __FILE__, function(){
    if ( ! wp_next_scheduled( 'smm_mp_cron_update_prices' ) ) {
        // every hour
        wp_schedule_event( time(), 'hourly', 'smm_mp_cron_update_prices' );
    }
    if ( ! wp_next_scheduled( 'smm_mp_cron_update_statuses' ) ) {
        // every 10 minutes
        wp_schedule_event( time(), 'ten_minutes', 'smm_mp_cron_update_statuses' );
    }

    // register custom recurrence for ten_minutes
    add_filter( 'cron_schedules', function( $schedules ){
        if ( ! isset( $schedules['ten_minutes'] ) ) {
            $schedules['ten_minutes'] = [
                'interval' => 10 * 60,
                'display'  => 'Every 10 Minutes'
            ];
        }
        return $schedules;
    } );
} );

register_deactivation_hook( __FILE__, function(){
    wp_clear_scheduled_hook( 'smm_mp_cron_update_prices' );
    wp_clear_scheduled_hook( 'smm_mp_cron_update_statuses' );
} );

/**
 * Ajax endpoints for admin actions
 */
add_action( 'wp_ajax_smm_mp_manual_sync_provider', function(){
    if ( ! current_user_can( 'manage_options' ) ) wp_send_json_error('no-permission');
    check_ajax_referer( 'smm_mp_nonce', 'nonce' );

    $provider_id = isset($_POST['provider_id']) ? absint($_POST['provider_id']) : 0;
    if ( ! $provider_id ) wp_send_json_error('invalid-provider');

    $sync = new SMM_MP_Sync();
    $res = $sync->sync_provider_services_to_products( $provider_id, true ); // true = create/update
    if ( is_wp_error( $res ) ) wp_send_json_error( $res->get_error_message() );
    wp_send_json_success( $res );
} );

add_action( 'wp_ajax_smm_mp_check_order_status', function(){
    if ( ! current_user_can( 'manage_woocommerce' ) ) wp_send_json_error('no-permission');
    check_ajax_referer( 'smm_mp_nonce', 'nonce' );

    $item_id = isset($_POST['item_id']) ? absint($_POST['item_id']) : 0;
    $order_id = isset($_POST['order_id']) ? absint($_POST['order_id']) : 0;
    if ( ! $item_id || ! $order_id ) wp_send_json_error('invalid-data');

    $sync = new SMM_MP_Sync();
    $res = $sync->check_single_item_status_ajax( $order_id, $item_id );
    if ( is_wp_error( $res ) ) wp_send_json_error( $res->get_error_message() );
    wp_send_json_success( $res );
} );

/**
 * Add custom cron schedule 'ten_minutes' (in case not added on activation)
 */
add_filter( 'cron_schedules', function( $schedules ){
    if ( ! isset( $schedules['ten_minutes'] ) ) {
        $schedules['ten_minutes'] = [
            'interval' => 10 * 60,
            'display'  => 'Every 10 Minutes'
        ];
    }
    return $schedules;
} );

/**
 * Hook scheduled events
 */
add_action( 'smm_mp_cron_update_prices', function(){
    $sync = new SMM_MP_Sync();
    $sync->cron_sync_all_providers_prices();
} );

add_action( 'smm_mp_cron_update_statuses', function(){
    $sync = new SMM_MP_Sync();
    $sync->cron_update_orders_status();
} );

/**
 * Enqueue admin JS
 */
add_action( 'admin_enqueue_scripts', function( $hook ){
    // load only on product edit and shop_order edit and providers CPT
    wp_register_script( 'smm-mp-admin-js', SMM_MP_PLUGIN_URL . 'assets/js/smm-wc-admin-status.js', ['jquery'], '1.0', true );
    wp_register_script( 'smm-mp-frontend-js', SMM_MP_PLUGIN_URL . 'assets/js/smm-wc-frontend.js', ['jquery'], '1.0', true );

    wp_localize_script( 'smm-mp-admin-js', 'SMM_MP_AJAX', [
        'ajax_url' => admin_url( 'admin-ajax.php' ),
        'nonce'    => wp_create_nonce( 'smm_mp_nonce' ),
        'i18n'     => [
            'loading' => 'جاري...',
            'error'   => 'حصل خطأ',
        ],
    ] );

    // enqueue on product edit / order edit / provider edit pages
    if ( in_array( $hook, [ 'post.php', 'post-new.php' ] ) ) {
        $post_type = get_post_type();
        if ( in_array( $post_type, [ 'product', 'shop_order', 'smm_provider' ] ) ) {
            wp_enqueue_script( 'smm-mp-admin-js' );
        }
    }
} );
