<?php
/**
 * Plugin Name: SMM API + WooCommerce Integration
 * Description: ربط خدمات SMM API مع ووكومرس: تحديد Service ID لكل منتج، جمع حقل الرابط والكمية، الطلب بعد الدفع، والتحقق من الحالة داخل لوحة الطلب.
 * Version: 1.1.0
 * Author: You
 */

if ( ! defined('ABSPATH') ) exit;

/** ========= الإعدادات العامة (API URL + API Key) ========= */
class SMM_API_WC_Settings {
    const OPTION_KEY = 'smm_api_wc_settings';

    public function __construct() {
        add_action('admin_menu', [$this, 'add_settings_page']);
        add_action('admin_init', [$this, 'register_settings']);
    }

    public function add_settings_page() {
        add_options_page('SMM API WC', 'SMM API WC', 'manage_options', 'smm-api-wc', [$this, 'settings_page_html']);
    }

    public function register_settings() {
        register_setting(self::OPTION_KEY, self::OPTION_KEY, [
            'type' => 'array',
            'sanitize_callback' => function($input){
                return [
                    'api_url' => esc_url_raw($input['api_url'] ?? ''),
                    'api_key' => sanitize_text_field($input['api_key'] ?? ''),
                ];
            },
        ]);

        add_settings_section('smm_api_wc_main', 'إعدادات API', '__return_false', self::OPTION_KEY);

        add_settings_field('api_url', 'API URL', function(){
            $opt = get_option(SMM_API_WC_Settings::OPTION_KEY);
            printf('<input type="url" name="%s[api_url]" value="%s" class="regular-text" placeholder="https://example.com/api/v2" required>',
                esc_attr(SMM_API_WC_Settings::OPTION_KEY),
                isset($opt['api_url']) ? esc_attr($opt['api_url']) : ''
            );
        }, self::OPTION_KEY, 'smm_api_wc_main');

        add_settings_field('api_key', 'API Key', function(){
            $opt = get_option(SMM_API_WC_Settings::OPTION_KEY);
            printf('<input type="text" name="%s[api_key]" value="%s" class="regular-text" placeholder="sk_live_xxx" required>',
                esc_attr(SMM_API_WC_Settings::OPTION_KEY),
                isset($opt['api_key']) ? esc_attr($opt['api_key']) : ''
            );
        }, self::OPTION_KEY, 'smm_api_wc_main');
    }

    public function settings_page_html() {
        if (!current_user_can('manage_options')) return; ?>
        <div class="wrap">
            <h1>إعدادات SMM API + WooCommerce</h1>
            <form method="post" action="options.php">
                <?php
                settings_fields(self::OPTION_KEY);
                do_settings_sections(self::OPTION_KEY);
                submit_button();
                ?>
            </form>
        </div>
        <?php
    }

    /** API Wrapper عام */
    public static function api_request(array $body) {
        $opt = get_option(self::OPTION_KEY);
        $api_url = $opt['api_url'] ?? '';
        $api_key = $opt['api_key'] ?? '';

        if (empty($api_url) || empty($api_key)) {
            return new WP_Error('smm_missing_config', 'يرجى ضبط API URL و API Key من الإعدادات.');
        }

        $body = array_merge(['key' => $api_key], $body);

        $resp = wp_remote_post($api_url, [
            'timeout' => 30,
            'headers' => [
                'Content-Type' => 'application/x-www-form-urlencoded; charset=utf-8',
                'User-Agent'   => 'WP-SMM-WC/1.1',
            ],
            'sslverify' => true, // لا تعطل SSL
            'body' => $body,
        ]);

        if (is_wp_error($resp)) return $resp;

        $code = wp_remote_retrieve_response_code($resp);
        $raw  = wp_remote_retrieve_body($resp);

        if ($code < 200 || $code >= 300) {
            return new WP_Error('smm_bad_status', 'فشل الاتصال بالـAPI. HTTP ' . $code, ['body' => $raw]);
        }

        $json = json_decode($raw, true);
        if (json_last_error() !== JSON_ERROR_NONE) {
            return new WP_Error('smm_json_error', 'استجابة غير صالحة من API.');
        }
        return $json;
    }

    /** عمليات الحالة */
    public static function api_status_single($api_order_id) {
        return self::api_request([
            'action' => 'status',
            'order'  => $api_order_id,
        ]);
    }

    public static function api_status_multi(array $ids) {
        $ids = array_filter(array_map('intval', $ids));
        if (empty($ids)) {
            return new WP_Error('smm_no_ids', 'لا يوجد أرقام طلبات خارجية.');
        }
        return self::api_request([
            'action' => 'status',
            'orders' => implode(',', $ids),
        ]);
    }
}
new SMM_API_WC_Settings();

/** ========= حقل Service ID للمنتج ========= */
class SMM_API_WC_Product {
    const META_SERVICE_ID = '_smm_service_id';

    public function __construct() {
        add_action('woocommerce_product_options_general_product_data', [$this, 'add_product_field']);
        add_action('woocommerce_process_product_meta', [$this, 'save_product_field']);
    }

    public function add_product_field() {
        echo '<div class="options_group">';
        woocommerce_wp_text_input([
            'id'          => self::META_SERVICE_ID,
            'label'       => 'SMM Service ID',
            'desc_tip'    => true,
            'description' => 'ضع رقم الخدمة كما في لوحة الـAPI.',
            'type'        => 'number',
            'custom_attributes' => ['min' => '1', 'step' => '1'],
        ]);
        echo '</div>';
    }

    public function save_product_field($post_id) {
        if (isset($_POST[self::META_SERVICE_ID])) {
            $val = absint($_POST[self::META_SERVICE_ID]);
            update_post_meta($post_id, self::META_SERVICE_ID, $val);
        }
    }
}
new SMM_API_WC_Product();

/** ========= عرض حقول "الرابط" و"الكمية" على صفحة المنتج ========= */
class SMM_API_WC_Frontend_Fields {
    const CART_KEY_LINK     = 'smm_link';
    const CART_KEY_QTY      = 'smm_quantity';

    public function __construct() {
        add_action('woocommerce_before_add_to_cart_button', [$this, 'render_fields']);
        add_filter('woocommerce_add_to_cart_validation',   [$this, 'validate_fields'], 10, 3);
        add_filter('woocommerce_add_cart_item_data',       [$this, 'save_cart_item_data'], 10, 2);
        add_filter('woocommerce_get_item_data',            [$this, 'display_cart_item_data'], 10, 2);
        add_action('woocommerce_checkout_create_order_line_item', [$this, 'add_to_order_items'], 10, 4);
    }

    private function product_has_service($product_id) {
        return (bool) get_post_meta($product_id, SMM_API_WC_Product::META_SERVICE_ID, true);
    }

    public function render_fields() {
        global $product;
        if (!$product || ! $this->product_has_service($product->get_id())) return; ?>
        <div class="smm-fields" style="margin:12px 0;">
            <p>
                <label for="smm_link"><strong>الرابط المطلوب</strong></label><br>
                <input type="url" id="smm_link" name="smm_link" required placeholder="https://example.com/test" style="width:100%;">
            </p>
            <p>
                <label for="smm_quantity"><strong>الكمية (API)</strong></label><br>
                <input type="number" id="smm_quantity" name="smm_quantity" required min="1" step="1" style="width:150px;">
            </p>
        </div>
        <?php
    }

    public function validate_fields($passed, $product_id, $quantity) {
        if (! $this->product_has_service($product_id)) return $passed;

        $link = isset($_POST['smm_link']) ? esc_url_raw($_POST['smm_link']) : '';
        $qty  = isset($_POST['smm_quantity']) ? absint($_POST['smm_quantity']) : 0;

        if (empty($link)) {
            wc_add_notice('الرجاء إدخال الرابط.', 'error');
            $passed = false;
        }
        if ($qty < 1) {
            wc_add_notice('الرجاء إدخال كمية صحيحة (1+).', 'error');
            $passed = false;
        }
        return $passed;
    }

    public function save_cart_item_data($cart_item_data, $product_id) {
        if (! $this->product_has_service($product_id)) return $cart_item_data;

        $cart_item_data[self::CART_KEY_LINK] = esc_url_raw($_POST['smm_link']);
        $cart_item_data[self::CART_KEY_QTY]  = absint($_POST['smm_quantity']);

        // لضمان تمييز العناصر:
        $cart_item_data['unique_key'] = md5(maybe_serialize([$cart_item_data[self::CART_KEY_LINK], $cart_item_data[self::CART_KEY_QTY], time()]));
        return $cart_item_data;
    }

    public function display_cart_item_data($item_data, $cart_item) {
        if (isset($cart_item[self::CART_KEY_LINK])) {
            $item_data[] = [
                'name'  => 'الرابط',
                'value' => esc_url($cart_item[self::CART_KEY_LINK]),
            ];
        }
        if (isset($cart_item[self::CART_KEY_QTY])) {
            $item_data[] = [
                'name'  => 'الكمية (API)',
                'value' => intval($cart_item[self::CART_KEY_QTY]),
            ];
        }
        return $item_data;
    }

    public function add_to_order_items($item, $cart_item_key, $values, $order) {
        if (isset($values[self::CART_KEY_LINK])) {
            $item->add_meta_data('SMM Link', esc_url_raw($values[self::CART_KEY_LINK]), true);
        }
        if (isset($values[self::CART_KEY_QTY])) {
            $item->add_meta_data('SMM Quantity', absint($values[self::CART_KEY_QTY]), true);
        }

        // خزن الـService ID على عنصر الطلب
        $service_id = get_post_meta($item->get_product_id(), SMM_API_WC_Product::META_SERVICE_ID, true);
        if ($service_id) {
            $item->add_meta_data('SMM Service ID', absint($service_id), true);
        }
    }
}
new SMM_API_WC_Frontend_Fields();

/** ========= تنفيذ الطلب بعد الدفع ========= */
class SMM_API_WC_After_Payment {
    public function __construct() {
        // يشتغل عند اكتمال الدفع
        add_action('woocommerce_payment_complete', [$this, 'send_api_orders']);
        // احتياط: عند تغيير الحالة إلى مكتمل
        add_action('woocommerce_order_status_completed', [$this, 'send_api_orders']);
    }

    public function send_api_orders($order_id) {
        $order = wc_get_order($order_id);
        if (!$order) return;

        foreach ($order->get_items() as $item_id => $item) {
            $service_id = $item->get_meta('SMM Service ID');
            $link       = $item->get_meta('SMM Link');
            $api_qty    = $item->get_meta('SMM Quantity');

            if (empty($service_id) || empty($link) || empty($api_qty)) {
                continue; // عنصر ليس له خدمة SMM
            }

            // هل سبق وإنشأنا طلب API لهذا العنصر؟
            $already = $item->get_meta('SMM API Order ID');
            if (!empty($already)) {
                continue;
            }

            // بناء الحمولة
            $payload = [
                'action'  => 'add',
                'service' => intval($service_id),
                'link'    => esc_url_raw($link),
                'quantity'=> intval($api_qty),
            ];

            $resp = SMM_API_WC_Settings::api_request($payload);

            if (is_wp_error($resp)) {
                $order->add_order_note('❌ فشل إنشاء طلب SMM لعنصر #' . $item_id . ': ' . $resp->get_error_message());
                continue;
            }

            // الاستجابة المتوقعة: { order: 12345, ... }
            if (!empty($resp['order'])) {
                $item->add_meta_data('SMM API Order ID', sanitize_text_field($resp['order']), true);
                $item->save();
                $order->add_order_note('✅ تم إنشاء طلب SMM لعنصر #' . $item_id . ' — رقم الطلب الخارجي: ' . $resp['order']);
            } else {
                $order->add_order_note('⚠️ استجابة غير متوقعة من SMM API لعنصر #' . $item_id . '.');
            }
        }
    }
}
new SMM_API_WC_After_Payment();

/** ========= ميتابوكس: التحقق من حالة طلبات SMM داخل صفحة الطلب ========= */
class SMM_API_WC_Admin_Status {
    const NONCE_ACTION = 'smm_wc_status_nonce';

    public function __construct() {
        add_action('add_meta_boxes', [$this, 'add_metabox']);
        add_action('admin_enqueue_scripts', [$this, 'enqueue_admin_assets']);
        add_action('wp_ajax_smm_wc_check_item_status', [$this, 'ajax_check_item_status']);
        add_action('wp_ajax_smm_wc_check_all_status',  [$this, 'ajax_check_all_status']);
    }

    public function add_metabox() {
        add_meta_box(
            'smm-wc-status-box',
            'SMM API — حالة الطلبات الخارجية',
            [$this, 'render_metabox'],
            'shop_order',
            'side',
            'default'
        );
    }

    public function enqueue_admin_assets($hook) {
        if ($hook !== 'post.php' && $hook !== 'post-new.php') return;
        if (get_post_type() !== 'shop_order') return;

        wp_register_script('smm-wc-admin-status', plugin_dir_url(__FILE__) . 'smm-wc-admin-status.js', ['jquery'], '1.0', true);
        wp_localize_script('smm-wc-admin-status', 'SMM_WC_STATUS', [
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce'    => wp_create_nonce(self::NONCE_ACTION),
            'i18n'     => [
                'loading' => 'جاري التحميل...',
                'done'    => 'تم التحقق.',
                'error'   => 'حدث خطأ.',
            ],
        ]);
        wp_enqueue_script('smm-wc-admin-status');
    }

    public function render_metabox($post) {
        $order = wc_get_order($post->ID);
        if (!$order) {
            echo '<p>لا يوجد طلب.</p>';
            return;
        }

        $items_with_api = [];
        foreach ($order->get_items() as $item_id => $item) {
            $api_order_id = $item->get_meta('SMM API Order ID');
            if (!empty($api_order_id)) {
                $items_with_api[$item_id] = [
                    'item'         => $item,
                    'api_order_id' => $api_order_id,
                ];
            }
        }

        if (empty($items_with_api)) {
            echo '<p>لا توجد عناصر مرتبطة برقم طلب خارجي بعد.</p>';
            return;
        }

        echo '<ul style="margin:0;padding-left:18px;">';
        foreach ($items_with_api as $item_id => $row) {
            $name = $row['item']->get_name();
            $api  = esc_html($row['api_order_id']);
            echo '<li style="margin-bottom:8px;">';
            echo '<div style="font-weight:600;">' . esc_html($name) . '</div>';
            echo '<div>API Order ID: <code>' . $api . '</code></div>';
            echo '<button type="button" class="button smm-check-one" data-item-id="'.esc_attr($item_id).'" data-api-order-id="'.esc_attr($api).'">تحقّق من الحالة</button>';
            echo '<div class="smm-status-output" id="smm-status-'.$item_id.'" style="margin-top:6px;font-size:12px;"></div>';
            echo '</li>';
        }
        echo '</ul>';

        echo '<hr />';
        echo '<button type="button" class="button button-primary smm-check-all" data-order-id="'.esc_attr($order->get_id()).'">تحقّق من حالة جميع العناصر</button>';
        echo '<div id="smm-status-all" style="margin-top:8px;font-size:12px;"></div>';
    }

    /** ========== AJAX: عنصر واحد ========== */
    public function ajax_check_item_status() {
        if (!current_user_can('manage_woocommerce')) wp_send_json_error(['message' => 'ليس لديك صلاحية.']);
        check_ajax_referer(self::NONCE_ACTION, 'nonce');

        $item_id     = isset($_POST['item_id']) ? absint($_POST['item_id']) : 0;
        $api_order_id= isset($_POST['api_order_id']) ? absint($_POST['api_order_id']) : 0;
        $order_id    = isset($_POST['order_id']) ? absint($_POST['order_id']) : 0;

        if (!$item_id || !$api_order_id || !$order_id) {
            wp_send_json_error(['message' => 'بيانات غير مكتملة.']);
        }

        $order = wc_get_order($order_id);
        if (!$order) wp_send_json_error(['message' => 'طلب غير موجود.']);

        $resp = SMM_API_WC_Settings::api_status_single($api_order_id);
        if (is_wp_error($resp)) {
            wp_send_json_error(['message' => $resp->get_error_message()]);
        }

        // الاستجابة المتوقعة: status, charge, remains, start_count, currency ...
        $this->maybe_add_order_note_from_status($order, $item_id, $api_order_id, $resp);

        wp_send_json_success(['data' => $resp]);
    }

    /** ========== AJAX: جميع العناصر (multiStatus) ========== */
    public function ajax_check_all_status() {
        if (!current_user_can('manage_woocommerce')) wp_send_json_error(['message' => 'ليس لديك صلاحية.']);
        check_ajax_referer(self::NONCE_ACTION, 'nonce');

        $order_id = isset($_POST['order_id']) ? absint($_POST['order_id']) : 0;
        if (!$order_id) wp_send_json_error(['message' => 'رقم الطلب غير صالح.']);

        $order = wc_get_order($order_id);
        if (!$order) wp_send_json_error(['message' => 'طلب غير موجود.']);

        $map_item_to_api = [];
        foreach ($order->get_items() as $item_id => $item) {
            $api_id = $item->get_meta('SMM API Order ID');
            if ($api_id) $map_item_to_api[$item_id] = absint($api_id);
        }
        if (empty($map_item_to_api)) {
            wp_send_json_error(['message' => 'لا توجد عناصر لها رقم طلب خارجي.']);
        }

        $resp = SMM_API_WC_Settings::api_status_multi(array_values($map_item_to_api));
        if (is_wp_error($resp)) {
            wp_send_json_error(['message' => $resp->get_error_message()]);
        }

        $results = $resp;

        foreach ($map_item_to_api as $item_id => $api_order_id) {
            $data = $this->extract_status_for_id($results, $api_order_id);
            if ($data) {
                $this->maybe_add_order_note_from_status($order, $item_id, $api_order_id, $data, false);
            }
        }

        $order->add_order_note('ℹ️ تم جلب حالات عناصر الطلب عبر multiStatus.');
        wp_send_json_success(['data' => $results]);
    }

    private function extract_status_for_id($results, $api_order_id) {
        if (isset($results[$api_order_id])) return $results[$api_order_id];
        if (is_array($results)) {
            foreach ($results as $k => $v) {
                if (is_array($v) && isset($v['order']) && intval($v['order']) === intval($api_order_id)) {
                    return $v;
                }
            }
        }
        return null;
    }

    private function maybe_add_order_note_from_status(WC_Order $order, $item_id, $api_order_id, $status_payload, $add_header = true) {
        $d = is_array($status_payload) ? $status_payload : (array) $status_payload;

        $parts = [];
        if (isset($d['status']))      $parts[] = 'الحالة: ' . $d['status'];
        if (isset($d['charge']))      $parts[] = 'الرسوم: ' . $d['charge'];
        if (isset($d['remains']))     $parts[] = 'المتبقي: ' . $d['remains'];
        if (isset($d['start_count'])) $parts[] = 'البدء: ' . $d['start_count'];
        if (isset($d['currency']))    $parts[] = 'العملة: ' . $d['currency'];

        $line = ($add_header ? '📦 عنصر #' . $item_id . ' — ' : '') . 'API Order ID ' . $api_order_id . ' → ' . (empty($parts) ? 'استجابة غير معروفة' : implode(' | ', $parts));
        $order->add_order_note($line);
    }
}
new SMM_API_WC_Admin_Status();
