<?php
if ( ! defined( 'ABSPATH' ) ) exit;

/**
 * Class SMM_MP_Sync
 * - Fetch services from providers (action=services)
 * - Update product prices (apply margin)
 * - Send API orders after payment (hooked from earlier code)
 * - Cron to update orders' statuses
 */
class SMM_MP_Sync {

    public function __construct() {
        // If you want to hook order-creation here (we already do from earlier file),
        // keep the earlier class that calls API after payment. We'll use this class for cron and helpers.
    }

    /**
     * Fetch services from a provider and create/update WP products or update prices.
     * $provider_id: CPT id
     * $create_products: if true, create product for services not present (simple product)
     */
    public function sync_provider_services_to_products( $provider_id, $create_products = false ) {
        $meta = SMM_MP_Provider::get_provider_meta( $provider_id );
        if ( ! $meta || empty( $meta['api_url'] ) || empty( $meta['api_key'] ) ) {
            return new WP_Error( 'no-config', 'Provider not configured.' );
        }

        $body = [
            'key' => $meta['api_key'],
            'action' => 'services'
        ];

        $resp = $this->remote_post( $meta['api_url'], $body );
        if ( is_wp_error( $resp ) ) return $resp;

        // Expecting array of services; each service may have id, name, rate/price/min,max...
        if ( ! is_array( $resp ) ) {
            return new WP_Error( 'bad-response', 'Unexpected response from provider.' );
        }

        $updated = 0;
        foreach ( $resp as $svc ) {
            // try common keys
            $service_id = isset( $svc['id'] ) ? $svc['id'] : ( isset($svc['service']) ? $svc['service'] : null );
            $service_name = isset( $svc['name'] ) ? $svc['name'] : ( isset($svc['title']) ? $svc['title'] : 'Service ' . $service_id );
            $rate = null;
            // possible fields for price/rate
            foreach ( ['rate','price','cost','min_price','price_from'] as $k ) {
                if ( isset( $svc[$k] ) ) { $rate = floatval( $svc[$k] ); break; }
            }

            if ( $service_id === null ) continue;

            // find existing product bound to this provider+service
            $existing = $this->find_product_by_provider_service( $provider_id, $service_id );

            if ( $existing ) {
                // update price if rate provided
                if ( $rate !== null ) {
                    $new_price = $this->apply_margin( $rate, $provider_id );
                    $product = wc_get_product( $existing );
                    if ( $product && floatval( $product->get_price() ) !== floatval( $new_price ) ) {
                        $product->set_regular_price( $new_price );
                        $product->save();
                        $updated++;
                    }
                }
            } else {
                if ( $create_products ) {
                    // create a simple product
                    $p = new WC_Product_Simple();
                    $p->set_name( $service_name );
                    $p->set_status( 'publish' );
                    $p->set_catalog_visibility( 'visible' );
                    $p->set_sku( 'smm-' . $provider_id . '-' . $service_id );
                    if ( $rate !== null ) {
                        $p->set_regular_price( $this->apply_margin( $rate, $provider_id ) );
                    } else {
                        $p->set_regular_price( 0 );
                    }
                    $new_id = $p->save();

                    // store provider & service meta
                    update_post_meta( $new_id, SMM_MP_Metabox::META_PROVIDER, absint( $provider_id ) );
                    update_post_meta( $new_id, SMM_MP_Metabox::META_SERVICE, sanitize_text_field( $service_id ) );

                    $updated++;
                } // end create_products
            }
        }

        return [ 'updated' => $updated ];
    }

    /**
     * Simple wrapper to post to provider
     */
    private function remote_post( $api_url, $body ) {
        $resp = wp_remote_post( $api_url, [
            'timeout' => 30,
            'body' => $body,
            'sslverify' => true,
            'headers' => [
                'Content-Type' => 'application/x-www-form-urlencoded; charset=utf-8',
                'User-Agent' => 'SMM-MP/1.0'
            ]
        ] );
        if ( is_wp_error( $resp ) ) return $resp;

        $code = wp_remote_retrieve_response_code( $resp );
        $raw  = wp_remote_retrieve_body( $resp );

        if ( $code < 200 || $code >= 300 ) {
            return new WP_Error( 'http_error', 'HTTP ' . $code . ' - ' . substr( $raw, 0, 200 ) );
        }

        $json = json_decode( $raw, true );
        if ( json_last_error() !== JSON_ERROR_NONE ) {
            // some providers return plain arrays; try to eval? No — return error
            return new WP_Error( 'json_error', 'Invalid JSON from provider.' );
        }

        return $json;
    }

    /**
     * Find product by stored provider+service meta
     */
    private function find_product_by_provider_service( $provider_id, $service_id ) {
        $args = [
            'post_type' => 'product',
            'posts_per_page' => 1,
            'meta_query' => [
                ['key' => SMM_MP_Metabox::META_PROVIDER, 'value' => $provider_id, 'compare' => '='],
                ['key' => SMM_MP_Metabox::META_SERVICE, 'value'  => $service_id, 'compare' => '='],
            ]
        ];
        $q = new WP_Query( $args );
        if ( $q->have_posts() ) {
            return $q->posts[0]->ID;
        }
        return false;
    }

    private function apply_margin( $base_price, $provider_id ) {
        $meta = SMM_MP_Provider::get_provider_meta( $provider_id );
        $margin = isset( $meta['margin'] ) ? floatval( $meta['margin'] ) : 0;
        $mult = 1 + ( $margin / 100 );
        return round( $base_price * $mult, 4 );
    }

    /* ---------- Cron actions ---------- */

    /**
     * Cron: sync all providers prices (called hourly)
     */
    public function cron_sync_all_providers_prices() {
        $providers = SMM_MP_Provider::get_active_providers();
        foreach ( $providers as $p ) {
            $this->sync_provider_services_to_products( $p->ID, false );
        }
    }

    /**
     * Cron: update orders statuses (called every 10 minutes)
     * Logic:
     *  - find orders in statuses: processing, on-hold (configurable later)
     *  - collect SMM API Order IDs, group by provider, call multiStatus when possible
     *  - add notes to orders and optionally update WC order status
     */
    public function cron_update_orders_status() {
        $args = [
            'limit' => -1,
            'status' => [ 'processing', 'on-hold' ],
        ];
        $orders = wc_get_orders( $args );
        if ( empty( $orders ) ) return;

        // map provider -> [ api_ids => [order_id,item_id,...] ]
        $map = [];

        foreach ( $orders as $order ) {
            foreach ( $order->get_items() as $item_id => $item ) {
                $api_order_id = $item->get_meta( 'SMM API Order ID' );
                $provider_id  = $item->get_meta( 'SMM Provider ID' );
                if ( $api_order_id && $provider_id ) {
                    $map[ $provider_id ][ $api_order_id ][] = [
                        'order_id' => $order->get_id(),
                        'item_id' => $item_id,
                    ];
                }
            }
        }

        foreach ( $map as $provider_id => $api_map ) {
            $provider_meta = SMM_MP_Provider::get_provider_meta( $provider_id );
            if ( ! $provider_meta ) continue;

            // for multiStatus we pass comma-separated list
            $ids = array_keys( $api_map );
            if ( empty( $ids ) ) continue;

            // call provider
            $body = [
                'key' => $provider_meta['api_key'],
                'action' => 'status',
                'orders' => implode( ',', $ids ),
            ];

            $resp = $this->remote_post( $provider_meta['api_url'], $body );
            if ( is_wp_error( $resp ) ) {
                // optionally log error
                continue;
            }

            // resp may be array keyed by api_order_id or list of items
            foreach ( $api_map as $api_order_id => $relations ) {
                $status_data = $this->extract_status_for_id( $resp, $api_order_id );
                if ( ! $status_data ) continue;

                foreach ( $relations as $r ) {
                    $order = wc_get_order( $r['order_id'] );
                    if ( ! $order ) continue;
                    $note = $this->status_to_note( $r['item_id'], $api_order_id, $status_data );
                    $order->add_order_note( $note );

                    // Optional: change WC order status if status_data indicates 'completed' or similar
                    if ( isset( $status_data['status'] ) ) {
                        $st = strtolower( $status_data['status'] );
                        if ( in_array( $st, ['completed','completed_success','delivered'] ) ) {
                            // mark order complete if not already
                            if ( $order->get_status() !== 'completed' ) {
                                $order->update_status( 'completed', 'Order marked completed based on SMM provider status.' );
                            }
                        }
                    }
                }
            }
        }
    }

    private function extract_status_for_id( $results, $api_order_id ) {
        if ( isset( $results[ $api_order_id ] ) ) return $results[ $api_order_id ];
        if ( is_array( $results ) ) {
            foreach ( $results as $k => $v ) {
                if ( is_array( $v ) && isset( $v['order'] ) && intval( $v['order'] ) === intval( $api_order_id ) ) {
                    return $v;
                }
            }
        }
        return null;
    }

    private function status_to_note( $item_id, $api_order_id, $d ) {
        $parts = [];
        if ( isset( $d['status'] ) ) $parts[] = 'الحالة: ' . $d['status'];
        if ( isset( $d['charge'] ) ) $parts[] = 'الرسوم: ' . $d['charge'];
        if ( isset( $d['remains'] ) ) $parts[] = 'المتبقي: ' . $d['remains'];
        if ( isset( $d['start_count'] ) ) $parts[] = 'البدء: ' . $d['start_count'];
        if ( isset( $d['currency'] ) ) $parts[] = 'العملة: ' . $d['currency'];

        return 'SMM Item #' . $item_id . ' — API Order ' . $api_order_id . ' → ' . implode( ' | ', $parts );
    }

    /* ---------- Ajax helper used by admin ajax single item check ---------- */

    public function check_single_item_status_ajax( $order_id, $item_id ) {
        $order = wc_get_order( $order_id );
        if ( ! $order ) return new WP_Error( 'no-order', 'Order not found' );
        $item = $order->get_item( $item_id );
        if ( ! $item ) return new WP_Error( 'no-item', 'Item not found' );
        $api_order_id = $item->get_meta( 'SMM API Order ID' );
        $provider_id  = $item->get_meta( 'SMM Provider ID' );
        if ( ! $api_order_id || ! $provider_id ) return new WP_Error( 'no-api', 'No API order id or provider' );
        $provider_meta = SMM_MP_Provider::get_provider_meta( $provider_id );
        if ( ! $provider_meta ) return new WP_Error( 'no-provider', 'Provider not found' );

        $body = [
            'key' => $provider_meta['api_key'],
            'action' => 'status',
            'order' => $api_order_id,
        ];

        $resp = $this->remote_post( $provider_meta['api_url'], $body );
        if ( is_wp_error( $resp ) ) return $resp;

        $status_data = $this->extract_status_for_id( $resp, $api_order_id ) ?: $resp;
        // add order note
        $order->add_order_note( $this->status_to_note( $item_id, $api_order_id, $status_data ) );
        return $status_data;
    }

    /* ---------- Helper to send an order to provider (used by previous after-payment handler) ---------- */

    public function send_order_to_provider( $provider_id, $payload ) {
        $provider_meta = SMM_MP_Provider::get_provider_meta( $provider_id );
        if ( ! $provider_meta ) return new WP_Error( 'no-provider', 'Provider not found' );
        $body = array_merge( [ 'key' => $provider_meta['api_key'] ], $payload );
        return $this->remote_post( $provider_meta['api_url'], $body );
    }
}
